/**
 * End-to-end sample (Node.js 18+)
 * token -> upload pdf -> add parties -> send invitations -> poll -> print download link
 */
import fs from "node:fs";
import fetch from "node-fetch";
import FormData from "form-data";

const BASE = process.env.FIRMATO_BASE_URL || "https://firmato.eu/sign/public/api/v1";
const CLIENT_ID = process.env.FIRMATO_CLIENT_ID;
const CLIENT_SECRET = process.env.FIRMATO_CLIENT_SECRET;

if (!CLIENT_ID || !CLIENT_SECRET) {
  console.error("Missing FIRMATO_CLIENT_ID / FIRMATO_CLIENT_SECRET");
  process.exit(1);
}

async function getToken() {
  const res = await fetch(`${BASE}/oauth/token`, {
    method: "POST",
    headers: {
      "Authorization": "Basic " + Buffer.from(`${CLIENT_ID}:${CLIENT_SECRET}`).toString("base64"),
      "Content-Type": "application/x-www-form-urlencoded",
    },
    body: new URLSearchParams({
      grant_type: "client_credentials",
      scope: "contracts:write contracts:read invites:send",
    }),
  });
  if (!res.ok) throw new Error(`Token error: ${res.status} ${await res.text()}`);
  return (await res.json()).access_token;
}

async function createContract(token, pdfPath, title) {
  const form = new FormData();
  form.append("file", fs.createReadStream(pdfPath));
  form.append("title", title);

  const res = await fetch(`${BASE}/contracts`, {
    method: "POST",
    headers: { "Authorization": `Bearer ${token}` },
    body: form,
  });
  if (!res.ok) throw new Error(`Create error: ${res.status} ${await res.text()}`);
  return await res.json();
}

async function addParties(token, contractId, parties) {
  const res = await fetch(`${BASE}/contracts/${contractId}/parties`, {
    method: "POST",
    headers: {
      "Authorization": `Bearer ${token}`,
      "Content-Type": "application/json",
    },
    body: JSON.stringify({ parties }),
  });
  if (!res.ok) throw new Error(`Parties error: ${res.status} ${await res.text()}`);
  return await res.json();
}

async function sendInvites(token, contractId) {
  const res = await fetch(`${BASE}/contracts/${contractId}/send`, {
    method: "POST",
    headers: { "Authorization": `Bearer ${token}` },
  });
  if (!res.ok) throw new Error(`Send error: ${res.status} ${await res.text()}`);
  return await res.json();
}

async function getStatus(token, contractId) {
  const res = await fetch(`${BASE}/contracts/${contractId}`, {
    method: "GET",
    headers: { "Authorization": `Bearer ${token}` },
  });
  if (!res.ok) throw new Error(`Status error: ${res.status} ${await res.text()}`);
  return await res.json();
}

(async () => {
  const token = await getToken();
  const created = await createContract(token, "./document.pdf", "My Document");
  console.log("Created:", created);

  const partiesRes = await addParties(token, created.contract_id, [
    { name: "Alice Doe", email: "alice@example.com" },
  ]);
  console.log("Parties:", partiesRes);

  const invites = await sendInvites(token, created.contract_id);
  console.log("Invites:", invites);

  const status = await getStatus(token, created.contract_id);
  console.log("Status:", status);

  console.log("Signer URL example:", status.links?.party_sign_urls?.[0]?.sign_url);
  console.log("Final stamped download URL (after all signatures):", status.links?.stamped_download_url);
})();
