"""End-to-end sample (Python 3.10+)
token -> upload pdf -> add parties -> send invitations -> poll -> print download link
"""
import os
import requests

BASE = os.getenv("FIRMATO_BASE_URL", "https://firmato.eu/sign/public/api/v1")
CLIENT_ID = os.getenv("FIRMATO_CLIENT_ID")
CLIENT_SECRET = os.getenv("FIRMATO_CLIENT_SECRET")

if not CLIENT_ID or not CLIENT_SECRET:
    raise SystemExit("Missing FIRMATO_CLIENT_ID / FIRMATO_CLIENT_SECRET")

def get_token() -> str:
    r = requests.post(
        f"{BASE}/oauth/token",
        auth=(CLIENT_ID, CLIENT_SECRET),
        data={"grant_type": "client_credentials", "scope": "contracts:write contracts:read invites:send"},
        timeout=30,
    )
    r.raise_for_status()
    return r.json()["access_token"]

def create_contract(token: str, pdf_path: str, title: str) -> dict:
    with open(pdf_path, "rb") as f:
        files = {"file": ("document.pdf", f, "application/pdf")}
        data = {"title": title}
        r = requests.post(f"{BASE}/contracts", headers={"Authorization": f"Bearer {token}"}, files=files, data=data, timeout=60)
        r.raise_for_status()
        return r.json()

def add_parties(token: str, contract_id: int, parties: list[dict]) -> dict:
    r = requests.post(
        f"{BASE}/contracts/{contract_id}/parties",
        headers={"Authorization": f"Bearer {token}", "Content-Type": "application/json"},
        json={"parties": parties},
        timeout=30,
    )
    r.raise_for_status()
    return r.json()

def send_invites(token: str, contract_id: int) -> dict:
    r = requests.post(f"{BASE}/contracts/{contract_id}/send", headers={"Authorization": f"Bearer {token}"}, timeout=30)
    r.raise_for_status()
    return r.json()

def get_status(token: str, contract_id: int) -> dict:
    r = requests.get(f"{BASE}/contracts/{contract_id}", headers={"Authorization": f"Bearer {token}"}, timeout=30)
    r.raise_for_status()
    return r.json()

if __name__ == "__main__":
    token = get_token()
    created = create_contract(token, "./document.pdf", "My Document")
    print("Created:", created)

    parties = add_parties(token, created["contract_id"], [{"name": "Alice Doe", "email": "alice@example.com"}])
    print("Parties:", parties)

    invites = send_invites(token, created["contract_id"])
    print("Invites:", invites)

    status = get_status(token, created["contract_id"])
    print("Status:", status)

    print("Signer URL example:", status.get("links", {}).get("party_sign_urls", [{}])[0].get("sign_url"))
    print("Final stamped download URL (after all signatures):", status.get("links", {}).get("stamped_download_url"))
